%{
Copyright (c) 2011, MIT.
Developed with the sponsorship of the Defense Advanced Research Projects
Agency (DARPA).

Permission is hereby granted, free of charge, to any person obtaining a copy
of this data, including any software or models in source or binary form, as
well as any drawings, specifications, and documentation (collectively "the
Data"), to deal in the Data without restriction, including without
limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Data, and to permit persons to whom
the Data is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE DATA OR
THE USE OR OTHER DEALINGS IN THE DATA.
%}

% Matt Fitzgerald
% Space Tug Transition Rules setup script

% DFC0 rules
% Rule 1: change cryo <--> biprop
% Rule 2: change amount of propellant
% DFC1 and DFC2 rules
% Rule 3: change cryo <--> biprop
% Rule 4: change amount of propellant
% Rule 5: change capability level
% DFC2 rules
% Rule 6: refuel

% $ Costs: max of (New Design Cost - Old Design Cost) or 1e-6 + a fee
% Fees:
% Rule 1: $150M
% Rule 2: $25M
% Rule 3: $75M
% Rule 4: $10M
% Rule 5: $75M
% Rule 6: $5M ----> $1M in the future

% Time costs - represented in matrices in MONTHS
% Rule 1: 7 months
% Rule 2: 5 months
% Rule 3: 5 months
% Rule 4: 4 months
% Rule 5: 4 months
% Rule 6: 2 months ----> 1 month in the future

% logic
% capability: change design num +/- 32*n for integer n within range
% propulsion: change design num +8 for biprop-->cryo, -8 for cryo-->biprop
% propellant: change design num within block of 8 designs (1-8, 9-16, etc)
% refuel: same as propellant but only UP TO itself

function [TcostPRESENT TtimePRESENT TcostFUTURE TtimeFUTURE] = SpaceTugTransMatrixGenerator(allCosts)

%% allCosts should be the cost matrix output of dataLoader

td = size(allCosts,1);  % should be 384

TcostPRESENT = cell(1,6);
TtimePRESENT = cell(1,6);
TcostFUTURE = cell(1,6);
TtimeFUTURE = cell(1,6);
for i = 1:6
	TcostPRESENT{i} = zeros(td,td);
	TtimePRESENT{i} = zeros(td,td);
    TcostFUTURE{i} = zeros(td,td);
	TtimeFUTURE{i} = zeros(td,td);
end

%% set costs to present and solve for those
costs = allCosts(:,1);
for i = 1:td
		% find allowable rule 1 and 3 executions
		set = 1:8;
        biprops0 = [set set+32 set+(32*2) set+(32*3)];
        biprops12 = [set+(32*4) set+(32*5) set+(32*6) set+(32*7) set+(32*8) set+(32*9) set+(32*10) set+(32*11)];
		if any(i == biprops0)
			TcostPRESENT{1}(i,i+8) = max(costs(i+8) - costs(i), 0)+ 150 ;
			TtimePRESENT{1}(i,i+8) = 7;
        end
        if any(i == biprops12)
            TcostPRESENT{3}(i,i+8) = max(costs(i+8) - costs(i), 0)+ 75 ;
            TtimePRESENT{3}(i,i+8) = 5;
        end
        if any(i == biprops0+8)
            TcostPRESENT{1}(i,i-8) = max(costs(i-8) - costs(i) , 0)+ 150 ;
            TtimePRESENT{1}(i,i-8) = 7;
        end
        if any(i == biprops12+8)
            TcostPRESENT{3}(i,i-8) = max(costs(i-8) - costs(i) , 0)+ 75 ;
            TtimePRESENT{3}(i,i-8) = 5;
        end
        
		% find proper block for rule 2/4
		remainder = mod(i,8);
        if remainder == 0
            remainder = 8;
        end
		for j = (i-remainder+1):(i-remainder+1+7)
			if i ~= j
                if i<129 % ie, it is a DFC0 design
                    TcostPRESENT{2}(i,j) = max(costs(j) - costs(i) , 0) + 25;
                    TtimePRESENT{2}(i,j) = 5;
                else
                    TcostPRESENT{4}(i,j) = max(costs(j) - costs(i) ,0) + 10;
                    TtimePRESENT{4}(i,j) = 4;
                end
			end
        end
        % do a similar loop for rule 6
        if i>256 % ie, it is a DFC2 design
            for j = (i-remainder+1):(i-remainder+8)
                TcostPRESENT{6}(i,j) = 5;
                TtimePRESENT{6}(i,j) = 2;
            end
        end
        

      % find the allowable rule 5 destinations
      if i>td/3 && i<=2*td/3
		possible1 = [i-3*32 i-2*32 i-32 i+32 i+2*32 i+3*32];
		nogoodSmall = possible1 <= td/3;
		nogoodLarge = possible1 > 2*td/3;
		nogood = (nogoodSmall+nogoodLarge)>0;
		possible1(nogood) = [];
		dest1 = possible1;
		% place entries
		for j = dest1
			TcostPRESENT{5}(i,j) = max(costs(j) - costs(i), 0) +75;
			TtimePRESENT{5}(i,j) = 4;
        end
      elseif i>2*td/3
        possible1 = [i-3*32 i-2*32 i-32 i+32 i+2*32 i+3*32];
		nogoodSmall = possible1 <= 2*td/3;
		nogoodLarge = possible1 > td;
		nogood = (nogoodSmall+nogoodLarge)>0;
		possible1(nogood) = [];
		dest1 = possible1;
		% place entries
		for j = dest1
			TcostPRESENT{5}(i,j) = max(costs(j) - costs(i), 0) +75;
			TtimePRESENT{5}(i,j) = 4;
        end
      end
end

%% repeat new loop with modified costs
costs = allCosts(:,9);
for i = 1:td
		% find allowable rule 1 and 3 executions
		set = 1:8;
        biprops0 = [set set+32 set+(32*2) set+(32*3)];
        biprops12 = [set+(32*4) set+(32*5) set+(32*6) set+(32*7) set+(32*8) set+(32*9) set+(32*10) set+(32*11)];
		if any(i == biprops0)
			TcostFUTURE{1}(i,i+8) = max(costs(i+8) - costs(i), 0)+ 150 ;
			TtimeFUTURE{1}(i,i+8) = 7;
        end
        if any(i == biprops12)
            TcostFUTURE{3}(i,i+8) = max(costs(i+8) - costs(i), 0)+ 75 ;
            TtimeFUTURE{3}(i,i+8) = 5;
        end
        if any(i == biprops0+8)
            TcostFUTURE{1}(i,i-8) = max(costs(i-8) - costs(i) , 0)+ 150 ;
            TtimeFUTURE{1}(i,i-8) = 7;
        end
        if any(i == biprops12+8)
            TcostFUTURE{3}(i,i-8) = max(costs(i-8) - costs(i) , 0)+ 75 ;
            TtimeFUTURE{3}(i,i-8) = 5;
        end
        
		% find proper block for rule 2/4
		remainder = mod(i,8);
        if remainder == 0
            remainder = 8;
        end
		for j = (i-remainder+1):(i-remainder+1+7)
			if i ~= j
                if i<129 % ie, it is a DFC0 design
                    TcostFUTURE{2}(i,j) = max(costs(j) - costs(i) , 0) + 25;
                    TtimeFUTURE{2}(i,j) = 5;
                else
                    TcostFUTURE{4}(i,j) = max(costs(j) - costs(i) ,0) + 10;
                    TtimeFUTURE{4}(i,j) = 4;
                end
            end
        end
        % do a similar loop for rule 6
        if i>256 % ie, it is a DFC2 design
            for j = (i-remainder+1):(i-remainder+8)
                TcostFUTURE{6}(i,j) = 1;
                TtimeFUTURE{6}(i,j) = 1;
            end

        end
        

      % find the allowable rule 5 destinations
      if i>td/3 && i<=2*td/3
		possible1 = [i-3*32 i-2*32 i-32 i+32 i+2*32 i+3*32];
		nogoodSmall = possible1 <= td/3;
		nogoodLarge = possible1 > 2*td/3;
		nogood = (nogoodSmall+nogoodLarge)>0;
		possible1(nogood) = [];
		dest1 = possible1;
		% place entries
		for j = dest1
			TcostFUTURE{5}(i,j) = max(costs(j) - costs(i), 0) +75;
			TtimeFUTURE{5}(i,j) = 4;
        end
      elseif i>2*td/3
        possible1 = [i-3*32 i-2*32 i-32 i+32 i+2*32 i+3*32];
		nogoodSmall = possible1 <= 2*td/3;
		nogoodLarge = possible1 > td;
		nogood = (nogoodSmall+nogoodLarge)>0;
		possible1(nogood) = [];
		dest1 = possible1;
		% place entries
		for j = dest1
			TcostFUTURE{5}(i,j) = max(costs(j) - costs(i), 0) +75;
			TtimeFUTURE{5}(i,j) = 4;
        end
      end
end



end